#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Video Renamer - GTK3 stable single-file
Features:
- GTK3 (PyGObject)
- Browse folder + Drag & Drop
- Sort-methods with checkboxes and move up/down
- ffprobe metadata extraction, broken file detection (moved to broken_files)
- Threaded processing + stop
- Animated progress (pulse) + fraction + "i of n" label
- Shows original name and new name in a table (new name updated after rename)
- Completion dialog when finished
- About dialog with author info
"""

from __future__ import annotations
import os
import json
import subprocess
import threading
import time
import sys
from datetime import datetime
from urllib.parse import unquote

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, Gdk, GLib

# Give the application a default icon name so individual windows also pick it up.
Gtk.Window.set_default_icon_name("gnome-todo")

# Supported video extensions
VIDEO_EXTS = {
    '.mp4', '.mkv', '.avi', '.mov', '.webm', '.flv', '.wmv',
    '.mpg', '.mpeg', '.3gp', '.ts', '.m4v', '.vob',
    '.ogv', '.f4v', '.rm', '.rmvb', '.m2ts', '.divx', '.asf'
}

FFPROBE_BASE = [
    'ffprobe', '-v', 'error',
    '-select_streams', 'v:0',
    '-show_entries', 'stream=width,height,duration',
    '-show_entries', 'format=duration',
    '-of', 'json'
]

def run_ffprobe(path, timeout=8):
    """Run ffprobe and return dict metadata or None."""
    try:
        proc = subprocess.run(FFPROBE_BASE + [path], capture_output=True, text=True, timeout=timeout)
        if proc.returncode != 0:
            return None
        data = json.loads(proc.stdout or "{}")
        dur = data.get("format", {}).get("duration")
        if dur is None:
            streams = data.get("streams", [])
            if streams:
                dur = streams[0].get("duration")
        if not dur:
            return None
        durf = float(dur)
        if durf <= 0:
            return None
        streams = data.get("streams", [])
        w = streams[0].get("width", 0) if streams else 0
        h = streams[0].get("height", 0) if streams else 0
        return {"duration": durf, "width": int(w or 0), "height": int(h or 0)}
    except Exception:
        return None

class AdvancedVideoRenamerApp:
    """
    The UI/controller. This class expects an existing Gtk.ApplicationWindow (so it works
    nicely with Gtk.Application and GNOME's session matching).
    """
    def __init__(self, window: Gtk.ApplicationWindow):
        # use the provided ApplicationWindow (created by Gtk.Application)
        self.window = window
        # try to help matching in some environments
        try:
            # set_wmclass exists but may be deprecated in some GTK versions; try anyway
            self.window.set_wmclass("video-renamer", "Video Renamer")
        except Exception:
            pass

        self.window.set_title("Video Renamer")
        self.window.set_default_size(1000, 650)
        self.window.connect("destroy", Gtk.main_quit)

        self.stop_event = threading.Event()
        self.worker_thread = None
        self.pulse_src = None

        self.build_ui()
        self.window.show_all()

    # ---------------- UI ----------------
    def build_ui(self):
        main_v = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8, margin=8)
        self.window.add(main_v)

        # Top: folder entry + browse + About button
        top_h = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6)
        main_v.pack_start(top_h, False, False, 0)

        self.folder_entry = Gtk.Entry()
        self.folder_entry.set_placeholder_text("Select folder or drag & drop folder here")
        self.folder_entry.set_hexpand(True)
        top_h.pack_start(self.folder_entry, True, True, 0)

        browse_btn = Gtk.Button(label="Browse Folder")
        browse_btn.connect("clicked", self.on_browse_clicked)
        top_h.pack_start(browse_btn, False, False, 0)

        # About button (shows author info)
        about_btn = Gtk.Button(label="About")
        about_btn.connect("clicked", self.on_about_clicked)
        top_h.pack_start(about_btn, False, False, 0)

        # Drag & Drop on folder_entry - روش ساده‌تر و مطمئن‌تر
        self.setup_drag_and_drop()

        # Options: base name, sort methods, order, progress label
        opts_h = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
        main_v.pack_start(opts_h, False, False, 0)

        # Left - base name + sort methods
        left_v = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6)
        opts_h.pack_start(left_v, False, False, 0)

        left_v.pack_start(Gtk.Label(label="Base name:"), False, False, 0)
        self.base_entry = Gtk.Entry()
        self.base_entry.set_text("My Video")
        left_v.pack_start(self.base_entry, False, False, 0)

        left_v.pack_start(Gtk.Label(label="Sort methods (priority top→bottom):"), False, False, 0)
        self.sort_store = Gtk.ListStore(bool, str)
        for m in ['duration', 'size', 'creation_date', 'modification_date', 'name', 'resolution']:
            self.sort_store.append([False, m])

        sort_tv = Gtk.TreeView(model=self.sort_store)
        renderer_toggle = Gtk.CellRendererToggle()
        renderer_toggle.connect("toggled", self.on_sort_toggled)
        col_toggle = Gtk.TreeViewColumn("Use", renderer_toggle, active=0)
        sort_tv.append_column(col_toggle)
        col_method = Gtk.TreeViewColumn("Method", Gtk.CellRendererText(), text=1)
        sort_tv.append_column(col_method)
        sort_tv.set_size_request(260, 160)
        left_v.pack_start(sort_tv, False, False, 0)

        # Move Up / Down
        mv_h = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6)
        up_btn = Gtk.Button(label="Move Up")
        up_btn.connect("clicked", self.on_move_up)
        down_btn = Gtk.Button(label="Move Down")
        down_btn.connect("clicked", self.on_move_down)
        mv_h.pack_start(up_btn, True, True, 0)
        mv_h.pack_start(down_btn, True, True, 0)
        left_v.pack_start(mv_h, False, False, 0)

        # Order radio
        order_h = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6)
        order_h.pack_start(Gtk.Label(label="Order:"), False, False, 0)
        self.asc_radio = Gtk.RadioButton.new_with_label_from_widget(None, "Ascending")
        self.desc_radio = Gtk.RadioButton.new_with_label_from_widget(self.asc_radio, "Descending")
        self.asc_radio.set_active(True)
        order_h.pack_start(self.asc_radio, False, False, 0)
        order_h.pack_start(self.desc_radio, False, False, 0)
        left_v.pack_start(order_h, False, False, 0)

        # Right - files list
        right_v = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6)
        opts_h.pack_start(right_v, True, True, 0)

        self.files_store = Gtk.ListStore(str, str, str)  # orig_path, new_name, status
        files_tv = Gtk.TreeView(model=self.files_store)
        renderer_orig = Gtk.CellRendererText()
        col_orig = Gtk.TreeViewColumn("Original path", renderer_orig, text=0)
        files_tv.append_column(col_orig)
        renderer_new = Gtk.CellRendererText()
        col_new = Gtk.TreeViewColumn("New name", renderer_new, text=1)
        files_tv.append_column(col_new)
        renderer_stat = Gtk.CellRendererText()
        col_stat = Gtk.TreeViewColumn("Status", renderer_stat, text=2)
        files_tv.append_column(col_stat)
        scrolled = Gtk.ScrolledWindow()
        scrolled.set_policy(Gtk.PolicyType.AUTOMATIC, Gtk.PolicyType.AUTOMATIC)
        scrolled.set_size_request(640, 260)
        scrolled.add(files_tv)
        right_v.pack_start(scrolled, True, True, 0)

        # Buttons: start / stop
        btn_h = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6)
        main_v.pack_start(btn_h, False, False, 0)
        self.start_btn = Gtk.Button(label="Start Renaming")
        self.start_btn.connect("clicked", self.on_start_clicked)
        self.stop_btn = Gtk.Button(label="Stop")
        self.stop_btn.connect("clicked", self.on_stop_clicked)
        self.stop_btn.set_sensitive(False)
        btn_h.pack_start(self.start_btn, False, False, 0)
        btn_h.pack_start(self.stop_btn, False, False, 0)

        # Progress + count
        prog_h = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6)
        main_v.pack_start(prog_h, False, False, 0)
        self.progress = Gtk.ProgressBar()
        self.progress.set_show_text(True)
        prog_h.pack_start(self.progress, True, True, 0)
        self.count_label = Gtk.Label(label="0 of 0")
        prog_h.pack_start(self.count_label, False, False, 0)

        # Log area
        log_frame = Gtk.Frame(label="Log")
        main_v.pack_start(log_frame, True, True, 0)
        vbox_log = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        log_frame.add(vbox_log)
        self.log_tv = Gtk.TextView()
        self.log_tv.set_editable(False)
        self.log_buf = self.log_tv.get_buffer()
        log_sc = Gtk.ScrolledWindow()
        log_sc.set_policy(Gtk.PolicyType.AUTOMATIC, Gtk.PolicyType.AUTOMATIC)
        log_sc.set_min_content_height(140)
        log_sc.add(self.log_tv)
        vbox_log.pack_start(log_sc, True, True, 0)

    def setup_drag_and_drop(self):
        """تنظیمات Drag & Drop با روش مطمئن‌تر"""
        # تنظیم هدف برای دریافت URIها
        targets = [
            Gtk.TargetEntry.new('text/uri-list', 0, 0),
            Gtk.TargetEntry.new('text/plain', 0, 1)
        ]
        
        # تنظیم ویجت برای دریافت drop
        self.folder_entry.drag_dest_set(
            Gtk.DestDefaults.ALL,
            targets,
            Gdk.DragAction.COPY
        )
        
        # متصل کردن سیگنال‌ها
        self.folder_entry.connect("drag-data-received", self.on_drag_data_received)
        self.folder_entry.connect("drag-motion", self.on_drag_motion)
        self.folder_entry.connect("drag-drop", self.on_drag_drop)

    def on_drag_motion(self, widget, drag_context, x, y, time):
        """هنگامی که موس روی ویجت حرکت می‌کند"""
        # بررسی اینکه آیا داده قابل قبول است
        if drag_context.list_targets():
            Gdk.drag_status(drag_context, Gdk.DragAction.COPY, time)
            return True
        return False

    def on_drag_drop(self, widget, drag_context, x, y, time):
        """هنگامی که آیتم drop می‌شود"""
        # درخواست داده‌ها
        targets = drag_context.list_targets()
        if targets:
            for target in targets:
                if target.name() in ['text/uri-list', 'text/plain']:
                    widget.drag_get_data(drag_context, target, time)
                    return True
        return False

    def on_drag_data_received(self, widget, drag_context, x, y, data, info, time):
        """پردازش داده‌های drop شده"""
        try:
            self.log("Drag & Drop data received")
            
            # پردازش URIها
            if data.get_uris():
                uris = data.get_uris()
                self.log(f"Received URIs: {uris}")
                
                paths = []
                for uri in uris:
                    if uri.startswith('file://'):
                        # حذف پیشوند و decode کردن
                        path = unquote(uri[7:])
                        # حذف کاراکترهای جدید خط
                        path = path.strip()
                        if os.path.exists(path):
                            paths.append(path)
                            self.log(f"Found path: {path}")

                if paths:
                    # پیدا کردن پوشه مناسب
                    target_folder = self.find_target_folder(paths)
                    if target_folder:
                        self.folder_entry.set_text(target_folder)
                        self.log(f"✅ Folder set via Drag & Drop: {target_folder}")
                    else:
                        self.log("❌ Could not determine target folder from dropped items")
                else:
                    self.log("❌ No valid paths found in dropped items")
            
            # پردازش متن ساده
            elif data.get_text():
                text = data.get_text().strip()
                self.log(f"Received text: {text}")
                if os.path.isdir(text):
                    self.folder_entry.set_text(text)
                    self.log(f"✅ Folder set via text drop: {text}")
            
            drag_context.finish(True, False, time)
            
        except Exception as e:
            self.log(f"❌ Drag & Drop error: {str(e)}")
            import traceback
            self.log(f"Traceback: {traceback.format_exc()}")
            drag_context.finish(False, False, time)

    def find_target_folder(self, paths):
        """پیدا کردن پوشه هدف از بین مسیرهای drop شده"""
        try:
            # اگر فقط یک مسیر داریم
            if len(paths) == 1:
                path = paths[0]
                if os.path.isdir(path):
                    return path
                elif os.path.isfile(path):
                    return os.path.dirname(path)
            
            # اگر چندین مسیر داریم
            else:
                # پیدا کردن پوشه والد مشترک
                dirs = []
                for path in paths:
                    if os.path.isdir(path):
                        dirs.append(path)
                    else:
                        dirs.append(os.path.dirname(path))
                
                if dirs:
                    common_dir = os.path.commonpath(dirs)
                    if os.path.isdir(common_dir):
                        return common_dir
                    
                # اگر پیدا کردن مشترک ممکن نبود، از اولین مسیر استفاده کن
                first_path = paths[0]
                if os.path.isfile(first_path):
                    return os.path.dirname(first_path)
                else:
                    return first_path
                    
        except Exception as e:
            self.log(f"Error finding target folder: {e}")
            # استفاده از اولین مسیر به عنوان fallback
            if paths:
                first_path = paths[0]
                if os.path.isfile(first_path):
                    return os.path.dirname(first_path)
                else:
                    return first_path
        
        return None

    # ---------------- Sort toggles & reorder ----------------
    def on_sort_toggled(self, cell, path):
        current = self.sort_store[path][0]
        self.sort_store[path][0] = not current

    def on_move_up(self, btn):
        sel = self.sort_store
        # find first selected row index
        idx = None
        for i, row in enumerate(self.sort_store):
            if row[0]:  # checked rows considered "selected" for move behavior; else could track selection
                idx = i
                break
        # if none specifically checked, get TreeView selection instead (more natural)
        if idx is None:
            return
        if idx <= 0:
            return
        # swap current row with previous row
        prev_iter = self.sort_store.get_iter(idx-1)
        cur_iter = self.sort_store.get_iter(idx)
        prev = list(self.sort_store[prev_iter])
        cur = list(self.sort_store[cur_iter])
        self.sort_store[prev_iter][:] = cur
        self.sort_store[cur_iter][:] = prev

    def on_move_down(self, btn):
        idx = None
        for i, row in enumerate(self.sort_store):
            if row[0]:
                idx = i
                break
        if idx is None:
            return
        if idx >= len(self.sort_store) - 1:
            return
        cur_iter = self.sort_store.get_iter(idx)
        next_iter = self.sort_store.get_iter(idx+1)
        cur = list(self.sort_store[cur_iter])
        nxt = list(self.sort_store[next_iter])
        self.sort_store[cur_iter][:] = nxt
        self.sort_store[next_iter][:] = cur

    # ---------------- Helpers ----------------
    def log(self, text):
        ts = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        line = f"[{ts}] {text}\n"
        GLib.idle_add(self._append_log, line)

    def _append_log(self, line):
        end = self.log_buf.get_end_iter()
        self.log_buf.insert(end, line)
        # autoscroll
        adj = self.log_tv.get_vadjustment()
        adj.set_value(adj.get_upper() - adj.get_page_size())
        return False

    def set_progress_fraction(self, frac):
        GLib.idle_add(self.progress.set_fraction, frac)
        # show percentage text
        GLib.idle_add(self.progress.set_text, f"{int(frac*100)}%")

    def set_count(self, cur, total):
        GLib.idle_add(self.count_label.set_text, f"{cur} of {total}")

    def add_file_row(self, fullpath, status="queued"):
        GLib.idle_add(self.files_store.append, [fullpath, "", status])

    def update_file_row(self, fullpath, new_name=None, status=None):
        def _upd():
            it = self.files_store.get_iter_first()
            while it:
                path = self.files_store.get_value(it, 0)
                if path == fullpath:
                    if new_name is not None:
                        self.files_store.set_value(it, 1, new_name)
                    if status is not None:
                        self.files_store.set_value(it, 2, status)
                    break
                it = self.files_store.iter_next(it)
            return False
        GLib.idle_add(_upd)

    # ---------------- Start / Stop UI actions ----------------
    def on_browse_clicked(self, btn):
        dlg = Gtk.FileChooserDialog(title="Select Folder", parent=self.window, action=Gtk.FileChooserAction.SELECT_FOLDER,
                                   buttons=(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL, "Select", Gtk.ResponseType.OK))
        response = dlg.run()
        if response == Gtk.ResponseType.OK:
            folder = dlg.get_filename()
            self.folder_entry.set_text(folder)
            self.log(f"Selected folder: {folder}")
        dlg.destroy()

    def on_start_clicked(self, btn):
        folder = self.folder_entry.get_text().strip()
        if not folder or not os.path.isdir(folder):
            self.show_error("Please select a valid folder.")
            return
        # gather methods in current priority order
        methods = []
        for row in self.sort_store:
            if row[0]:
                methods.append(row[1])
        if not methods:
            methods = ['name']
        order = 'asc' if self.asc_radio.get_active() else 'desc'
        base_name = self.base_entry.get_text().strip() or "My Video"

        self.log(f"Start: folder={folder}, methods={methods}, order={order}, base='{base_name}'")

        self.files_store.clear()
        self.stop_event.clear()
        self.start_btn.set_sensitive(False)
        self.stop_btn.set_sensitive(True)

        # start pulse
        self.start_pulse()

        self.worker_thread = threading.Thread(target=self.process_videos, args=(folder, methods, order, base_name), daemon=True)
        self.worker_thread.start()

    def on_stop_clicked(self, btn):
        self.log("Stop requested by user.")
        self.stop_event.set()
        self.stop_btn.set_sensitive(False)

    # ---------------- About dialog ----------------
    def on_about_clicked(self, btn):
        about = Gtk.AboutDialog(transient_for=self.window, modal=True)
        about.set_program_name("Video Renamer")
        about.set_version("1.0")
        about.set_comments("GTK3-based batch video renamer with metadata sorting.")
        about.set_authors(["Pourdaryaei"])
        about.set_website("https://pourdaryaei.ir")
        # use the installed icon name if available
        try:
            about.set_logo_icon_name("gnome-todo")
        except Exception:
            pass
        about.run()
        about.destroy()

    # ---------------- Pulse animation ----------------
    def start_pulse(self):
        if self.pulse_src is None:
            def _pulse():
                try:
                    self.progress.pulse()
                except Exception:
                    pass
                return True
            self.pulse_src = GLib.timeout_add(120, _pulse)

    def stop_pulse(self):
        if self.pulse_src is not None:
            try:
                GLib.source_remove(self.pulse_src)
            except Exception:
                pass
            self.pulse_src = None

    # ---------------- Core processing ----------------
    def process_videos(self, folder, methods, order, base_name):
        try:
            # Stage 1: collect files recursively
            all_files = []
            for root, dirs, files in os.walk(folder):
                for f in files:
                    if os.path.splitext(f)[1].lower() in VIDEO_EXTS:
                        all_files.append(os.path.join(root, f))
            total_found = len(all_files)
            self.log(f"Files found: {total_found}")
            if total_found == 0:
                self.stop_pulse()
                GLib.idle_add(self.finish_processing)
                return

            # populate table
            for p in all_files:
                self.add_file_row(p, "queued")

            # Stage 2: extract metadata
            valid = []
            broken = []
            for i, p in enumerate(all_files):
                if self.stop_event.is_set():
                    self.log("Stopped during metadata extraction.")
                    self.stop_pulse()
                    GLib.idle_add(self.finish_processing)
                    return
                self.set_count(i+1, total_found)
                # set progress approx
                self.set_progress_fraction(i / total_found)
                self.update_file_row(p, status="analyzing")
                meta = run_ffprobe(p, timeout=8)
                if meta:
                    metaobj = {
                        'path': p,
                        'duration': meta.get('duration'),
                        'size': os.path.getsize(p),
                        'creation_date': os.path.getctime(p),
                        'modification_date': os.path.getmtime(p),
                        'name': os.path.basename(p),
                        'width': meta.get('width'),
                        'height': meta.get('height'),
                    }
                    valid.append(metaobj)
                    self.update_file_row(p, status="valid")
                else:
                    broken.append(p)
                    self.update_file_row(p, status="broken")
                    self.log(f"Broken file: {os.path.basename(p)}")
                time.sleep(0.01)

            # Stage 3: move broken
            if broken:
                broken_dir = os.path.join(folder, "broken_files")
                if not os.path.exists(broken_dir):
                    try:
                        os.makedirs(broken_dir, exist_ok=True)
                        self.log(f"Created broken_files folder: {broken_dir}")
                    except Exception as e:
                        self.log(f"Could not create broken_files dir: {e}")
                for b in broken:
                    try:
                        dest = os.path.join(broken_dir, os.path.basename(b))
                        base, ext = os.path.splitext(dest)
                        c = 1
                        while os.path.exists(dest):
                            dest = f"{base}_{c}{ext}"
                            c += 1
                        os.rename(b, dest)
                        self.log(f"Moved broken file: {os.path.basename(b)} → {dest}")
                    except Exception as e:
                        self.log(f"Failed moving broken file {b}: {e}")

            # Stage 4: sort valid files
            self.log("Sorting files...")
            sorted_valid = self.sort_files(valid, methods, order)

            # Stage 5: rename
            total_valid = len(sorted_valid)
            pad = max(2, len(str(total_valid)))
            for i, fd in enumerate(sorted_valid):
                if self.stop_event.is_set():
                    self.log("Stopped during renaming.")
                    self.stop_pulse()
                    GLib.idle_add(self.finish_processing)
                    return
                old = fd['path']
                ext = os.path.splitext(old)[1]
                new_name = f"{base_name} {str(i+1).zfill(pad)}{ext}"
                # Keep files in their original directory
                original_dir = os.path.dirname(old)
                new_path = os.path.join(original_dir, new_name)
                # avoid overwrite
                basep, extp = os.path.splitext(new_path)
                c = 1
                final_new = new_path
                while os.path.exists(final_new):
                    final_new = f"{basep}_{c}{extp}"
                    c += 1
                try:
                    os.rename(old, final_new)
                    self.log(f"✅ {os.path.basename(old)} → {os.path.basename(final_new)} ({fd.get('duration'):.1f}s, {fd.get('size')} bytes)")
                    # update UI: set new name & status
                    self.update_file_row(old, new_name=os.path.basename(final_new), status="renamed")
                except Exception as e:
                    self.log(f"Failed rename {old}: {e}")
                    self.update_file_row(old, status="failed")
                # update progress and count
                self.set_count(i+1, total_valid)
                frac = (i+1) / total_valid if total_valid > 0 else 1.0
                self.set_progress_fraction(frac)
                time.sleep(0.02)

            # Stage 6: summary & finish
            self.log("\n🎉 === Operation Summary ===")
            self.log(f"Total files found: {total_found}")
            self.log(f"Files renamed: {total_valid}")
            self.log(f"Broken files: {len(broken)}")
            self.log(f"Sort method: {', '.join(methods)} ({order})")
            self.stop_pulse()
            GLib.idle_add(self.show_done_dialog)
        except Exception as e:
            self.log(f"Unexpected error: {e}")
        finally:
            GLib.idle_add(self.finish_processing)

    def sort_files(self, files, methods, order):
        rev = (order == 'desc')
        def key(f):
            k = []
            for m in methods:
                if m == 'duration':
                    k.append(f.get('duration', 0) or 0)
                elif m == 'size':
                    k.append(f.get('size', 0) or 0)
                elif m == 'creation_date':
                    k.append(f.get('creation_date', 0) or 0)
                elif m == 'modification_date':
                    k.append(f.get('modification_date', 0) or 0)
                elif m == 'name':
                    k.append(f.get('name', '').lower())
                elif m == 'resolution':
                    w = f.get('width') or 0
                    h = f.get('height') or 0
                    k.append(w*h)
                else:
                    k.append(0)
            return tuple(k)
        try:
            return sorted(files, key=key, reverse=rev)
        except Exception as e:
            self.log(f"Sort error: {e}")
            return files

    def finish_processing(self):
        self.start_btn.set_sensitive(True)
        self.stop_btn.set_sensitive(False)
        self.stop_pulse()
        self.set_progress_fraction(0.0)
        self.set_count(0,0)
        return False

    def show_done_dialog(self):
        dlg = Gtk.MessageDialog(parent=self.window, flags=0, message_type=Gtk.MessageType.INFO,
                                buttons=Gtk.ButtonsType.OK, text="Processing completed")        
        dlg.run()
        dlg.destroy()
        return False

    def show_error(self, message):
        dlg = Gtk.MessageDialog(parent=self.window, flags=0, message_type=Gtk.MessageType.ERROR,
                                buttons=Gtk.ButtonsType.CLOSE, text="Error")
        dlg.format_secondary_text(message)
        dlg.run()
        dlg.destroy()
        return False

# ----------------- Application wrapper -----------------
class VideoRenamerApplication(Gtk.Application):
    def __init__(self):
        # application id should be reverse-domain; helps GNOME match windows to desktop files
        super().__init__(application_id="com.pourdaryaei.VideoRenamer", flags=0)

    def do_activate(self):
        # Create ApplicationWindow (ties window to application - helps GNOME shell)
        win = Gtk.ApplicationWindow(application=self)
        # set the window icon name to match installed icon
        try:
            win.set_icon_name("gnome-todo")
        except Exception:
            pass
        app_ui = AdvancedVideoRenamerApp(win)
        win.show_all()

    def do_startup(self):
        Gtk.Application.do_startup(self)

def main():
    app = VideoRenamerApplication()
    # run the GTK application
    exit_status = app.run(sys.argv)
    sys.exit(exit_status)

if __name__ == "__main__":
    main()
